<?php

namespace App\Http\Controllers;

use App\Jobs\ProcessExcelChunkJob;
use App\Models\UploadedFile;
use App\Services\ExcelImportService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class UploadController extends Controller
{
    protected $excelService;

    public function __construct(ExcelImportService $excelService)
    {
        $this->excelService = $excelService;
    }

    /**
     * Show the upload page
     */
    public function index()
    {
        $uploadedFiles = UploadedFile::orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return view('upload.index', compact('uploadedFiles'));
    }

    /**
     * Handle file upload
     */
    public function uploadExcel(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,csv|max:102400', // Max 100MB
        ]);

        try {
            $file = $request->file('file');
            $originalFilename = $file->getClientOriginalName();
            $filename = Str::uuid() . '.' . $file->getClientOriginalExtension();

            // Store file
            $filePath = $file->storeAs('uploads', $filename, 'local');
            $fullPath = storage_path('app/' . $filePath);

            // Get total rows
            $totalRows = $this->excelService->getTotalRows($fullPath);

            // Get header mapping
            $headerMapping = $this->excelService->getHeaders($fullPath);

            // Create uploaded file record
            $uploadedFile = UploadedFile::create([
                'filename' => $filename,
                'original_filename' => $originalFilename,
                'status' => UploadedFile::STATUS_PENDING,
                'total_rows' => $totalRows,
                'processed_rows' => 0,
                'file_path' => $fullPath,
            ]);

            // Dispatch first chunk job
            $chunkSize = config('excel_columns.chunk_size', 1000);
            ProcessExcelChunkJob::dispatch(
                $uploadedFile->id,
                2, // Start from row 2 (skip header)
                $chunkSize,
                $headerMapping
            );

            // Update status to processing
            $uploadedFile->update(['status' => UploadedFile::STATUS_PROCESSING]);

            return response()->json([
                'success' => true,
                'message' => 'File uploaded successfully. Processing started.',
                'upload_id' => $uploadedFile->id,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error uploading file: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get upload status
     */
    public function getUploadStatus($id)
    {
        $uploadedFile = UploadedFile::findOrFail($id);

        return response()->json([
            'id' => $uploadedFile->id,
            'status' => $uploadedFile->status,
            'total_rows' => $uploadedFile->total_rows,
            'processed_rows' => $uploadedFile->processed_rows,
            'progress_percentage' => round($uploadedFile->progress_percentage, 2),
            'error_message' => $uploadedFile->error_message,
            'created_at' => $uploadedFile->created_at->format('Y-m-d H:i:s'),
        ]);
    }
}

